/** @file   sound.cpp
 * @brief   Implementation of Sound - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
 
#include "sound.h"
#include <fstream>

using std::vector;
using std::string;
using std::ifstream;
using std::map;

namespace eng2d {


//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

int Sound::requestedVoices = 16;
bool Sound::soundsOn = true;
int Sound::masterVolume = 128;
int Sound::defaultFrequency = 1000;
int Sound::updateCounter = 0;
int Sound::minimumReplayDelay = -1;

vector<SAMPLE*> Sound::m_samples;
vector<SoundSource*> Sound::m_soundsources;
map<string, SAMPLE*> Sound::m_samplesByName;
vector< int > Sound::m_lastPlayTimes;


///
/// Installing and removing
/// =======================

/** Installs the sound interface
 */
int Sound::install()
{
  reserve_voices( requestedVoices, 0 );
  return install_sound( DIGI_AUTODETECT, MIDI_AUTODETECT, 0 );
}



/** Removes the sound interface
 */
void Sound::remove()
{
  remove_sound();
}





///
/// Managing and playing the samples.
/// =================================

/** Initializes the samples
 */
int Sound::initSamples( const string& sfile )
{
  if ( readSamples( sfile ) != 0 ) {
    return -1;
  }
  return 0;
}



/** Cleans up the samples
 */
void Sound::flushSamples()
{
  for (int i=0; i < m_samples.size(); i++) {
    if ( m_samples.at(i) != 0 ) {
      destroy_sample( m_samples.at(i) );
      m_samples.at(i) = 0;
    }
  }
  m_samples.clear();
}



/** Plays the sample. This method uses the default values
 */
void Sound::playSample( int index, bool loop )
{
  if ( index >= m_samples.size() ) {
    return;
  }
  
  // If player has turned the sounds off
  if ( soundsOn == false ) {
    return;
  }
  
  if ( m_samples.at(index) == 0 ) {
    return;
  }
  
  // Handle the minimum replay delay.
  if ( minimumReplayDelay >= 0 ) {
    int timeDifference = updateCounter - m_lastPlayTimes.at( index );
    if ( timeDifference < minimumReplayDelay ) {
      return;
    }
    m_lastPlayTimes.at( index ) = updateCounter;
  }
  
  play_sample( m_samples.at(index), 128, 128, defaultFrequency,
               static_cast<int>(loop) );
}



/** Plays the unloopped sample that is located at given distance
 */
void Sound::playDistanceSample(int index, float distance)
{
  if ( distance > 600 ) {
    return;
  }
  
  distance /= 180;
  distance += 0.3;
  
  float vol = 128 / distance;
  if ( vol < 20 ) {
    vol = 20;
  } else if ( vol > 150 ) {
    vol = 150;
  }
  play_sample( m_samples.at(index), static_cast<int>(vol), 128,
               defaultFrequency, 0 );
}



/** Stops the requested sample
 */
void Sound::stopSample( int index )
{
  stop_sample( m_samples.at( index ) );
}



/** Stops all the samples
 */
void Sound::stopAllSamples()
{
  for (int i=0; i < m_samples.size(); i++ ) {
    stop_sample( m_samples.at(i) );
  }
}



///
/// The positional sounds
/// =====================

/** Creates and plays new ambience sound
 */
void Sound::createSoundSource( SoundSource* pSource )
{
  map<string, SAMPLE*>::iterator it = m_samplesByName.find( pSource->name() );
  if ( it == m_samplesByName.end() ) {
    // We couldn't find the sample matching the name of this soundsource.
    delete pSource;
    return;
  }
  pSource->setData( it->second );
  pSource->turnOn();
  m_soundsources.push_back( pSource );
}



/** Removes the given soundsource
 */
void Sound::removeSoundSource( SoundSource* pSource )
{
  vector<SoundSource*>::iterator it = m_soundsources.begin();
  while ( it != m_soundsources.end() ) {
    if ( (*it) == pSource ) {
      delete pSource;
      m_soundsources.erase( it );
      return;
    }
    it++;
  }
}



/** Updates the soundsources
 */
void Sound::update( const Vec2D& rListener )
{
  updateCounter += 1;
  for ( int i=0; i < m_soundsources.size(); i++ ) {
    m_soundsources.at(i)->update( rListener );
  }
}



/** Removes all the positional samples.
 */
void Sound::clearSoundSources()
{
  for ( int i=0; i < m_soundsources.size(); i++ ) {
    if ( m_soundsources.at(i) != 0 ) {
      delete m_soundsources.at(i);
      m_soundsources.at(i) = 0;
    }
  }
  m_soundsources.clear();
}



/** Returns the number of soundsources there are.
 */
int Sound::numberOfSoundSources()
{
  return m_soundsources.size();
}



/** Tells if the given soundsource is in the tables
 */
bool Sound::soundSourceExists( SoundSource* pSS )
{
  for ( int i=0; i < m_soundsources.size(); i++ ) {
    if ( m_soundsources.at(i) == pSS ) {
      return true;
    }
  }
  return false;
}




//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************



//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

/** Reads the samples
 */
int Sound::readSamples( const string& sfile )
{
  ifstream fin( sfile.c_str() );
  if ( !fin ) {
    return -1;
  }
  
  string tmp;
  while ( true ) {
    if ( fin.eof() == true ) {
      fin.close();
      return -1;
    }
    fin >> tmp;
    if ( tmp == "[END]" ) {
      fin.close();
      return 0;
      
    } else if ( tmp == "<wewantwar_soundsamples>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      fin.ignore( 4096, '\n' );
      
    }
  }
  
  // We've found the beginning tag.
  int ret = 0;
  while ( true ) {
    if ( fin.eof() == true ) {
      fin.close();
      return -1;
    }
    fin >> tmp;
    if ( tmp == "#" ) {
      fin.ignore( 4096, '\n' );
      
    } else if ( tmp == "</wewantwar_soundsamples>" ) {
      fin.close();
      return ret;
      
    } else if ( tmp == "<sample>" ) {
      string sampleName = "";
      string sampleFile = "";
      while ( true ) {
        if ( fin.eof() == true ) {
          fin.close();
          return -1;
        }
        fin >> tmp;
        if ( tmp == "#" ) {
          fin.ignore( 4096, '\n' );
        } else if ( tmp == "name:" ) {
          fin >> sampleName;
        } else if ( tmp == "file:" ) {
          fin >> sampleFile;
        } else if ( tmp == "</sample>" ) {
          break;
        }
      }
      // Load the sample unless it's an empty sample.
      SAMPLE* pS = 0;
      if ( sampleName != "EMPTY_SAMPLE" ) {
        pS = load_sample( sampleFile.c_str() );

        if ( pS == 0 ) {
          ret = -1;
        }
      }
      m_samples.push_back( pS );
      m_samplesByName[sampleName] = pS;
      m_lastPlayTimes.push_back( 0 );
    }
  }
  fin.close();
  return ret;
}


} // end of namespace
